<?php

use Symfony\Component\Finder\Finder;

class Madhouse_ThemeKitchen_Services_StaticConfigurationService
{
    protected $model;
    protected $themes;

    /**
     * @TODO
     */
    public static function newInstance()
    {
        return new self(
            new Madhouse_ThemeKitchen_Models_Settings(),
            WebThemes::newInstance()
        );
    }

    public function __construct($model, $themeManager)
    {
        $this->model = $model;
        $this->themes = $themeManager;
    }

    public function findConfigurationBy()
    {
        // Retrieve @ database & decode.
        $configuration = $this->model->getConfiguration();

        return $configuration;
    }

    public function findBlocks($filter = null)
    {
        $configuration = $this->findConfigurationBy();

        $customWidgets = array();

        $finder = new Finder();
        foreach ($configuration->get("$.blocks.path") as $path) {
            $absolutePath = $this->themes->getCurrentThemePath() . '/' . $path;
            $finder->in($absolutePath);
        }

        try {
            // Get info for each match files.
            foreach ($finder->files() as $file) {
                $customWidgets[] = $this->getBlockInfos($file->getRealPath());
            }
        } catch (\LogicException $e) {
            // Should not happen but happens when $.blocks.path is not defined or empty.
            return array();
        }

        // Sort by names.
        usort($customWidgets, function ($a, $b) {
            return strnatcmp($a["s_name"], $b["s_name"]);
        });

        // Return list of all custom blocks.
        return $customWidgets;
    }

    public function findBlock($blockSlug, $blocks = null)
    {
        if (is_null($blocks)) {
            $blocks = $this->findBlocks();
        }

        // Filter blocks.
        $blocks = array_filter(
            $blocks,
            function ($block) use ($blockSlug) {
                return ($block["s_slug"] === $blockSlug);
            }
        );

        if (empty($blocks)) {
            return null;
        }

        return array_shift($blocks);
    }

    public function getBlockInfos($path)
    {
        $s_info = file_get_contents($path);
        $info   = array(
            "s_slug" => "",
            "s_name" => "",
            "s_description" => "",
            "s_path" => $path,
            "widget_hooks" => array(),
        );

        if (preg_match('|Block Slug:([^\r\t\n]*)|i', $s_info, $match)) {
            $info["s_slug"] = trim($match[1]);
        }

        if (preg_match('|Block Name:([^\r\t\n]*)|i', $s_info, $match)) {
            $info["s_name"] = trim($match[1]);
        }

        if (preg_match('|Description:([^\r\t\n]*)|i', $s_info, $match)) {
            $info["s_description"] = trim($match[1]);
        }

        if (preg_match('|Widget Hooks:([^\r\t\n]*)|i', $s_info, $match)) {
            $info["widget_hooks"] = preg_split('/\s*,\s*/', trim($match[1]));
        }

        return $info;
    }

    public function findSocialNetworks()
    {
        $configuration = $this->findConfigurationBy();
        return $configuration->get("$.social_networks");
    }

    public function findWidgetHooks()
    {
        $configuration = $this->findConfigurationBy();
        return array_merge(
            $configuration->get("$.widget_hooks"),
            array_map(
                function ($v) {
                    return array(
                        "s_slug" => $v["s_internal_name"],
                        "b_filters_enabled" => false,
                        "s_group" => "static_page"
                    );
                },
                Page::newInstance()->listAll(0, 1)
            )
        );
    }

    public function findPages()
    {
        $configuration = $this->findConfigurationBy();
        return $configuration->get("$.pages");
    }

    /**
     * Update the configuration with given configuration.
     *
     * @param  Madhouse_ThemeKitchen_Settings2 $configuration.
     *
     * @return Madhouse_ThemeKitchen_Settings2
     */
    public function updateConfiguration($configuration)
    {
        // Set configuration @ database.
        $this->model->set("configuration", $configuration);

        // Return an updated configuration.
        return $this->findConfigurationBy();
    }

    /**
     * Add a new widget hook to the configuration.
     *
     * @param array $widgetHook
     */
    public function addWidgetHook($configuration, $widgetHook)
    {
        if (!$configuration instanceof Madhouse_ThemeKitchen_Settings2) {
            throw new \InvalidArgumentException(
                "'configuration' must be a valid Madhouse_ThemeKitchen_Settings2."
            );
        }

        if (!$this->checkWidgetHook($widgetHook)) {
            throw new \InvalidArgumentException(
                "Widget hook is not valid."
            );
        }

        if ($this->hasWidgetHook($configuration, $widgetHook["s_slug"])) {
            // Already exist, do nothing.
            return $configuration;
        }

        // Get widget hooks.
        $widgetHooks = $configuration->get("$.widget_hooks");

        // Push the widget hook in the array.
        $widgetHooks[] = $widgetHook;

        // Sort, remove duplicate and set the widget hook in the object.
        $configuration->set("$.widget_hooks", array_values($widgetHooks));

        // Update configuration @ database.
        return $this->updateConfiguration($configuration);
    }

    /**
     * Does the configuration has a widget hook with slug $widgetHookSlug?
     *
     * @param  Madhouse_ThemeKitchen_Settings2  $configuration
     * @param  string  $widgetHookSlug
     *
     * @return boolean
     */
    public function hasWidgetHook($configuration, $widgetHookSlug)
    {
        // Get widget hooks.
        $widgetHooks = $this->findWidgetHooks();

        // Get all widget slugs.
        $widgetHookSlugs = array_map(
            function ($widgetHook) {
                return $widgetHook["s_slug"];
            },
            $widgetHooks
        );

        // Widget hook exist?
        return in_array($widgetHookSlug, $widgetHookSlugs);
    }

    /**
     * Utility function to validate a widget hook type and structure.
     *
     * @param  array $widgetHook
     *
     * @return boolean
     */
    protected function checkWidgetHook($widgetHook)
    {
        return (is_array($widgetHook) && isset($widgetHook["s_slug"]) && !empty($widgetHook["s_slug"]));
    }

    /**
     * Add a new widget hook to the configuration.
     *
     * @param array $widgetHook
     *
     * @return Madhouse_ThemeKitchen_Configuration
     */
    public function addSocialNetwork($configuration, $socialNetwork)
    {
        if (!$configuration instanceof Madhouse_ThemeKitchen_Settings2) {
            throw new \InvalidArgumentException(
                "'configuration' must be a valid Madhouse_ThemeKitchen_Settings2."
            );
        }

        if (!$this->checkSocialNetwork($socialNetwork)) {
            throw new \InvalidArgumentException(
                "Social network is not valid."
            );
        }

        if ($this->hasSocialNetwork($configuration, $socialNetwork["s_slug"])) {
            // Already exist, do nothing.
            return $configuration;
        }

        // Get widget hooks.
        $socialNetworks = $configuration->get("$.social_networks");

        // Push the widget hook in the array.
        $socialNetworks[] = $socialNetwork;

        // Sort, remove duplicate and set the widget hook in the object.
        $configuration->set("$.social_networks", array_values($socialNetworks));

        // Update configuration @ database.
        return $this->updateConfiguration($configuration);
    }

    /**
     * Does the configuration has a social network with slug $socialNetworkSlug?
     *
     * @param  Madhouse_ThemeKitchen_Settings2  $configuration
     * @param  string                           $socialNetworkSlug
     *
     * @return boolean
     */
    public function hasSocialNetwork($configuration, $socialNetworkSlug)
    {
        // Get widget hooks.
        $socialNetworks = $this->findSocialNetworks();

        // Get all widget slugs.
        $socialNetworkSlugs = array_map(
            function ($socialNetwork) {
                return $socialNetwork["s_slug"];
            },
            $socialNetworks
        );

        // Social network exist?
        return in_array($socialNetworkSlug, $socialNetworkSlugs);
    }

    /**
     * Utility function to validate a social network type and structure.
     *
     * @param  array $widgetHook
     *
     * @return boolean
     */
    protected function checkSocialNetwork($socialNetwork)
    {
        return (is_array($socialNetwork) && isset($socialNetwork["s_slug"]) && !empty($socialNetwork["s_slug"]));
    }
}
