<?php

class Madhouse_ThemeKitchen_Package
{

    /**
     * Current version of this plugin.
     *
     * @since 1.0.1
     */
    const CURRENT_VERSION   = "1.3.0";

    /**
     * Install the theme.
     *
     * Merge the settings and configuration from theme if exists, or install
     * default settings and configuration if not. It will do nothing if settings
     * exists in database.
     *
     * @return void
     */
    public static function install()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.0.0";

        if ($oldVersion == "") {
            /**
             * Configuration.
             */
            $settingsService = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();
            $defaultConfiguration = json_decode(file_get_contents(self::getPath("assets/model/configuration.json")), true);
            $themeFile = WebThemes::newInstance()->getCurrentThemePath() . "/assets/model/themekitchen-configuration.json";
            if (file_exists($themeFile)) {
                $themeConfiguration = json_decode(file_get_contents($themeFile), true);
                if (is_null($themeConfiguration)) {
                    // @TODO : log / warning ?
                    $themeConfiguration = array();
                }
            } else {
                $themeConfiguration = array();
            }

            // Merge configurations.
            $configurationToInstall = array_merge_recursive($defaultConfiguration, $themeConfiguration);
            $configurationToInstall["blocks"]["path"] = array_values(array_unique($configurationToInstall["blocks"]["path"], SORT_REGULAR));
            $configurationToInstall["social_networks"] = array_values(array_unique($configurationToInstall["social_networks"], SORT_REGULAR));
            $configurationToInstall["widget_hooks"] = array_values(array_unique($configurationToInstall["widget_hooks"], SORT_REGULAR));

            // Install configuration.
            $model->set("configuration", $configurationToInstall);

            /**
             * Settings.
             */
            $currentSettings = $settingsService->findSettingsBy();
            if ($currentSettings->isEmpty()) {
                $defaultSettings = self::getPath("assets/model/install.sql");
                $themeSettings = WebThemes::newInstance()->getCurrentThemePath() . "/assets/model/themekitchen-install.sql";

                $settingsFile = null;
                if (file_exists($themeSettings)) {
                    $settingsFile = $themeSettings;
                } elseif (file_exists($defaultSettings)) {
                    $settingsFile = $defaultSettings;
                }

                // If settingsFile found.
                if ($settingsFile && file_exists($settingsFile)) {
                    // Import the default (from theme if exists of from themekitchen) settings SQL file.
                    self::importSQL($settingsFile);
                }
            }

            // Sets the version.
            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    /**
     * Upgrade to v1.0.1
     *
     * @return void
     *
     * @since  1.0.1
     */
    public static function upgrade101()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.0.1";

        if (version_compare($oldVersion, $newVersion) < 0) {
            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    /**
     * Upgrade to v1.0.2
     *
     * @return void
     *
     * @since  1.0.2
     */
    public static function upgrade102()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.0.2";

        if (version_compare($oldVersion, $newVersion) < 0) {
            // Retrieve the current configuration.
            $configurationService = Madhouse_ThemeKitchen_Services_StaticConfigurationService::newInstance();
            $configurationObject = $configurationService->findConfigurationBy();

            // Set all 'b_filters_enabled' to their default value.
            foreach ($configurationObject->get("widget_hooks") as $index => $widgetHook) {
                if (!isset($widgetHook["b_filters_enabled"])) {
                    // Set all to default except 'item_sidebar'.
                    $default = false;
                    if ($widgetHook["s_slug"] === "item_sidebar") {
                        $default = true;
                    }

                    // Set default to the object.
                    $configurationObject->set(
                        array("widget_hooks", $index, "b_filters_enabled"),
                        $default
                    );
                }
            }

            // Update configuration.
            $configurationService->updateConfiguration($configurationObject);

            // Set version to $newVersion.
            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    /**
     * Upgrade themekitchen.
     *
     * @return void
     *
     * @since  1.0.3
     */
    public static function upgrade103()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.0.3";

        if (version_compare($oldVersion, $newVersion) < 0) {
            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    /**
     * Upgrade to v1.0.4
     *
     * @return void
     *
     * @since  1.0.4
     */
    public static function upgrade104()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.0.4";

        if (version_compare($oldVersion, $newVersion) < 0) {
            $settingsService = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();
            $module = Madhouse_ThemeKitchen_Services_ModuleManager::newInstance()->getModule("layout");

            $currentSettings = $settingsService->findSettingsBy();

            $currentSettings->set("$.layout.i_homepage_item_popular_number", 12);
            $currentSettings->set("$.layout.i_homepage_item_premium_number", 12);

            $currentSettings->set("$.layout.i_search_item_premium_number", 9);

            $currentSettings->set("$.layout.i_item_post_display_location", 1);
            $currentSettings->set("$.layout.i_item_post_display_zip", 1);
            $currentSettings->set("$.layout.i_item_post_display_city_area", 1);
            $currentSettings->set("$.layout.i_item_post_display_address", 1);

            $layoutSettings = $currentSettings->getSubset("$.layout");

            // Update stuff @ database.
            $model->updateModule(
                $module,
                $layoutSettings,
                $layoutSettings
            );

            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    /**
     * Upgrade themekitchen.
     *
     * @return void
     *
     * @since  1.0.5
     */
    public static function upgrade105()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.0.5";

        if (version_compare($oldVersion, $newVersion) < 0) {
            $settingsService = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();
            $module = Madhouse_ThemeKitchen_Services_ModuleManager::newInstance()->getModule("layout");

            // Get current settings.
            $currentSettings = $settingsService->findSettingsBy();

            // Set the new setting.
            $currentSettings->set("$.layout.i_item_pictures_mode", 0);

            // Update stuff @ database.
            $layoutSettings = $currentSettings->getSubset("$.layout");
            $model->updateModule(
                $module,
                $layoutSettings,
                $layoutSettings
            );

            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    /*
     * Upgrade themekitchen.
     *
     * @return void
     *
     * @since  1.0.6
     */
    public static function upgrade106()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.0.6";

        if (version_compare($oldVersion, $newVersion) < 0) {
            $settingsService = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();
            $module = Madhouse_ThemeKitchen_Services_ModuleManager::newInstance()->getModule("layout");

            $currentSettings = $settingsService->findSettingsBy();

            $currentSettings->set("$.layout.i_search_sidebar_position", 0);
            $currentSettings->set("$.layout.i_search_slider", 1);
            $currentSettings->set("$.layout.i_register_user_phone", 0);
            $currentSettings->set("$.layout.i_register_user_type", 0);
            $currentSettings->set("$.layout.i_register_user_website", 0);
            $currentSettings->set("$.layout.i_register_user_country", 0);
            $currentSettings->set("$.layout.i_register_user_region", 0);
            $currentSettings->set("$.layout.i_register_user_city", 0);
            $layoutSettings  = $currentSettings->getSubset("$.layout");

            // Update stuff @ database.
            $model->updateModule(
                $module,
                $layoutSettings,
                $layoutSettings
            );

            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    public static function upgrade107()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.0.7";

        if (version_compare($oldVersion, $newVersion) < 0) {

            /*
             * Update static configuration.
             */

            // Get the configuration service.
            $configurationService = Madhouse_ThemeKitchen_Services_StaticConfigurationService::newInstance();

            // Retrieve the configuration from database.
            $currentConfiguration = $configurationService->findConfigurationBy();

            // Add the widget hooks.
            $currentConfiguration = $configurationService->addWidgetHook(
                $currentConfiguration,
                array(
                    "s_slug" => "item_top",
                    "b_allow_static_pages" => true,
                )
            );
            $currentConfiguration = $configurationService->addWidgetHook(
                $currentConfiguration,
                array(
                    "s_slug" => "item_bottom",
                    "b_allow_static_pages" => true,
                )
            );

            /*
             * Add new settings (default).
             */

            // Get the settings service.
            $settingsService = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();

            // Retrieve settings from database.
            $currentSettings = $settingsService->findSettingsBy();

            // Set new settings.
            $currentSettings->set("$.layout.i_register_user_location_style", 0);
            $currentSettings->set("$.layout.i_item_quick_view", 1);

            // Get the subset of layout.
            $layoutSettings  = $currentSettings->getSubset("$.layout");

            // Get the module.
            $module = Madhouse_ThemeKitchen_Services_ModuleManager::newInstance()->getModule("layout");

            // Update stuff @ database.
            $model->updateModule(
                $module,
                $layoutSettings,
                $layoutSettings
            );

            /*
             * Set the new version @ database.
             */
            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    public static function upgrade110()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.1.0";

        if (version_compare($oldVersion, $newVersion) < 0) {

            // Get the settings service.
            $settingsService = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();

            // Retrieve settings from database.
            $currentSettings = $settingsService->findSettingsBy();

            // Set new settings.
            $currentSettings->set("$.layout.i_search_location_style", 0);
            $currentSettings->set("$.layout.i_item_post_display_category", 0);

            // Update settings @ database.
            $settingsService->updateSettings($currentSettings);

            /*
             * Set the new version @ database.
             */
            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    public static function upgrade120()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.2.0";

        if (version_compare($oldVersion, $newVersion) < 0) {

            /*
             * Update static configuration.
             */

            // Get the configuration service.
            $configurationService = Madhouse_ThemeKitchen_Services_StaticConfigurationService::newInstance();

            // Retrieve the configuration from database.
            $currentConfiguration = $configurationService->findConfigurationBy();

            // Add the widget hooks.
            $currentConfiguration = $configurationService->addSocialNetwork(
                $currentConfiguration,
                array(
                    "s_slug" => "instagram",
                    "s_name" => "Instagram",
                    "b_metatags" => false,
                )
            );

            /*
             * Add new settings (default).
             */

            // Get the settings service.
            $settingsService = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();

            // Retrieve settings from database.
            $currentSettings = $settingsService->findSettingsBy();

            // Set new settings.
            $currentSettings->set("$.layout.i_search_category_display", 0);
            $currentSettings->set("$.layout.i_search_location_select_display", 0);

            // Update settings @ database.
            $settingsService->updateSettings($currentSettings);

            /*
             * Set the new version @ database.
             */
            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    public static function upgrade130()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");
        $newVersion = "1.3.0";

        if (version_compare($oldVersion, $newVersion) < 0) {

            /*
             * Add new settings (default).
             */

            // Get the settings service.
            $settingsService = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();

            // Retrieve settings from database.
            $currentSettings = $settingsService->findSettingsBy();

            // Set new settings.
            $currentSettings->set("$.layout.i_item_post_ajax_photos", 0);
            $currentSettings->set("$.layout.i_search_mode_live", 1);

            // Update settings @ database.
            $settingsService->updateSettings($currentSettings);

            /*
             * Set the new version @ database.
             */
            $model->set("themekitchen_version", $newVersion);
            osc_reset_preferences();
        }
    }

    /**
     * Upgrade themekitchen.
     *
     * @return void
     *
     * @since 1.0.1
     */
    public static function init()
    {
        self::upgrade();
    }


     /**
     * Init themekitchen.
     *
     * @return void
     *
     * @since 1.0.1
     */
    public static function upgrade()
    {
        $model = new Madhouse_ThemeKitchen_Models_Settings();
        $oldVersion = $model->get("themekitchen_version");

        if ($oldVersion != "") {
            self::upgrade101();
            self::upgrade102();
            self::upgrade103();
            self::upgrade104();
            self::upgrade105();
            self::upgrade106();
            self::upgrade107();
            self::upgrade110();
            self::upgrade120();
            self::upgrade130();
        }
    }

    /**
     * Get theme slug
     *
     * @return string  Theme Slug
     * @since 1.0.0
     */
    public static function getThemeSlug()
    {
        return osc_current_web_theme();
    }

    /**
     * Get relative path
     *
     * @return string Relative path
     * @since 1.0.0
     */
    public static function getRelativePath($file = null)
    {
        // Get the package path.
        $packagePath = dirname(dirname(dirname(dirname(__FILE__))));

        // Sanitize path for Windows (grr...).
        $packagePath = preg_replace('|/+|','/', str_replace('\\','/', $packagePath));

        // Compute the relative path to themekitchen.
        $relativePath = preg_replace('#^.*/(oc-content.*)/?$#', '$1', $packagePath);

        // Append given $file, if any.
        if (!is_null($file)) {
            $relativePath .= '/' . $file;
        }

        return $relativePath;
    }

    /**
     * Gets the absolute path of a file.
     *
     * @since 1.00
     */
    public static function getPath($file)
    {
        return implode('/', array(osc_base_path(), self::getRelativePath($file)));
    }

    /**
     * Gets the URL of a file from the root of Osclass.
     *
     * @since 1.00
     */
    public static function getURL($file)
    {
        return osc_base_url() . self::getRelativePath($file);
    }

    /**
     * Imports an SQL file into the Osclass database.
     *
     * Really useful when installing a plugin to create its model schema,
     * or delete its schema on uninstall. Avoid to have an install/uninstall
     * method in every plugin data-access object.
     *
     * @param   $path absolute file path to the SQL file to import.
     *
     * @return  void
     *
     * @throws  Exception if the import fails.
     *
     * @see     Madhouse_Utils_Models::import
     */
    public static function importSQL($path)
    {
        // Try to import it. Throws Exception if failure.
        $conn = DBConnectionClass::newInstance()->getOsclassDb();
        $dao = new DBCommandClass($conn);

        // // Get the content of the file.
        $sql = file_get_contents($path);

        // Strip comments.
        // Makes DAO go wazoo.
        $sql = preg_replace('/\-\-.*$/m', '', $sql);

        if (! $dao->importSQL($sql)) {
            throw new Exception("Import failed with: '" . $dao->getErrorDesc() . "'");
        }
    }
}
