<?php

class Madhouse_ThemeKitchen_Controllers_Admin extends AdminSecBaseModel
{
    /**
     * Settings service for ThemeKitchen.
     *
     * @var Madhouse_ThemeKitchen_Services_SettingsService
     */
    protected $settings;

    public function __construct()
    {
        // Set settings service.
        $this->configuration = Madhouse_ThemeKitchen_Services_StaticConfigurationService::newInstance();
        $this->modules = Madhouse_ThemeKitchen_Services_ModuleManager::newInstance();
        $this->settings = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();
    }

    /**
     * Controller dispatch method.
     *
     * @return void.
     */
    public function doModel()
    {
        if (mtk_is_settings_post_page()) {

            // Save new settings action.
            $this->updateSettingsAction();

        } elseif(mtk_settings_delete_resource_post_page()) {

            // Delete requested resource action.
            $this->deleteResourceAction();

        } elseif (mtk_is_settings_page()) {

            // Get the requested module.
            $submoduleId = Params::getParam("submodule_id");

            // Show settings for this module.
            $this->showSettingsAction($submoduleId);

        } else {
            // Do nothing;
        }
    }

    /**
     * Show settings page for the requested module.
     *
     * @param  string $moduleId
     *
     * @return void.
     *
     * @since  1.1.0
     */
    public function showSettingsAction($submoduleId)
    {
        // Export the submodule.
        try {
            // Get the module.
            $submodule = $this->modules->getModule($submoduleId);
        } catch (Exception $e) {
            // Redirect to the right section.
            osc_add_flash_error_message(__("Module not found!", "themekitchen"), "admin");
            osc_redirect_to(mtk_settings_url());
        }
        $this->_exportVariableToView("submodule", $submodule);

        // Export the available custom widgets.
        $this->_exportVariableToView("custom_blocks", $this->configuration->findBlocks());

        // Export the supported social networks.
        $this->_exportVariableToView("social_networks", $this->configuration->findSocialNetworks());

        // Export the pages list.
        // Those are not static pages but more like routes.
        $this->_exportVariableToView("pages", $this->configuration->findPages());

        // Export the categories as a tree.
        $this->_exportVariableToView("categories", Category::newInstance()->toTree());

        // Export the static pages.
        // Those are pages created in admin and stored in oc_t_page SQL table.
        $this->_exportVariableToView("static_pages", Page::newInstance()->listAll(0));

        // Export the supported widget locations.
        $this->_exportVariableToView("widgets", $this->configuration->findWidgetHooks());

        // Run a hook before the page.
        osc_run_hook("mdh_themekitchen_show_settings", $submodule);
    }

    /**
     * Update the settings action.
     *
     * @return void.
     *
     * @since  1.1.0
     */
    public function updateSettingsAction()
    {
        $this->checkDemo(__("Sorry, you can't update theme settings on a demo website.", "themekitchen"));

        $moduleId = Params::getParam("submodule_id");

        // Get current section & postedSettings
        $settingsSent = json_decode(Params::getParam("_settings", false, false), true);

        try {
            // Get the module.
            $module = $this->modules->getModule($moduleId);
        } catch (Exception $e) {
            // Redirect to the right section.
            osc_add_flash_error_message(__("Module not found!", "themekitchen"), "admin");
            osc_redirect_to(osc_get_http_referer());
        }

        // Retrieve settings.
        $currentSettings = $this->settings->findSettingsBy(
            array("module" => $module)
        );

        // Update module settings with new sent ones.
        //$currentSettings->set("$." . $moduleId, $settingsSent[$moduleId]);

        try {
            // Update stuff @ database.
            $this->settings->updateModule(
                $module,
                $currentSettings,
                new Madhouse_ThemeKitchen_Settings2(
                    (isset($settingsSent[$moduleId])) ?$settingsSent[$moduleId] : array()
                )
            );

            // Redirect to the right section.
            osc_add_flash_ok_message(__("Successfully updated settings!", "themekitchen"), "admin");
            osc_redirect_to(mtk_settings_url($moduleId));

        } catch (Exception $e) {
            osc_add_flash_error_message(
                sprintf(__("Something went wrong during the update: '%s'!", "themekitchen"), $e->getMessage()),
                "admin"
            );
            osc_redirect_to(osc_get_http_referer());
        }
    }

    /**
     * Delete a resource from settings action.
     *
     * @return void.
     *
     * @since  1.1.0
     */
    public function deleteResourceAction()
    {
        $this->checkDemo(__("Sorry, you can't update theme settings on a demo website.", "themekitchen"));

        $resourceSection = Params::getParam("resource_section");
        $resourceName = Params::getParam("resource_name");
        $resourceLocale = Params::getParam("resource_locale");

        // Retrieve settings for that module.
        $currentSettings = $this->settings->findSettingsBy();

        // Get resource.
        $currentResource = $currentSettings->get($resourceSection, $resourceName, $resourceLocale);
        if (!$currentResource || !is_array($currentResource)) {
            osc_add_flash_error_message(
                sprintf(__("Something went wrong during the update: '%s'!", "themekitchen"), ""),
                "admin"
            );
            osc_redirect_to(osc_get_http_referer());
        }

        // Remove the resource.
        $currentSettings->set($currentSettings->buildQuery($resourceSection, $resourceName, $resourceLocale), array());

        try {
            // Update stuff @ database.
            $this->settings->updateSettings($currentSettings);

            $files = array();
            foreach ($currentResource["formats"] as $format) {
                $formatPath = ABS_PATH . '/' . $currentResource["s_path"] . '/' . $currentResource["s_name"] . "_" . $format . "." .  $currentResource["s_ext"];
                $files[] = $formatPath;
            }
            $files[] = ABS_PATH . '/' . $currentResource["s_path"] . '/' . $currentResource["s_name"] . "." . $currentResource["s_ext"];
            foreach ($files as $file) {
                if (file_exists($file) && is_file($file)) {
                    unlink($file);
                }
            }

            // Redirect to the right section.
            osc_add_flash_ok_message(__("Successfully updated settings!", "themekitchen"), "admin");
            osc_redirect_to(osc_get_http_referer());

        } catch (Exception $e) {
            osc_add_flash_error_message(
                sprintf(__("Something went wrong during the update: '%s'!", "themekitchen"), $e->getMessage()),
                "admin"
            );
            osc_redirect_to(osc_get_http_referer());
        }
    }

    /**
     * Fails and redirects, with a message $message, if the website has DEMO enabled.
     *
     * @param  string $message
     *
     * @return void.
     *
     * @since  1.1.0
     */
    protected function checkDemo($message)
    {
        if (defined('DEMO') && DEMO == true) {
            osc_add_flash_warning_message($message, 'admin');
            $this->redirectTo(osc_get_http_referer());
        }
    }
}
