<?php

use Leafo\ScssPhp\Compiler as ScssCompiler;

class Madhouse_ThemeKitchen_Modules_DesignModule extends Madhouse_ThemeKitchen_ModuleBase
{
    /**
     * Static configuration service.
     *
     * @var Madhouse_
     */
    protected $configuration;

    /**
     * Theme manager.
     *
     * @var Themes
     */
    protected $themes;

    public function __construct($configurationService, $themeManager)
    {
        $this->configuration = $configurationService;
        $this->themes = $themeManager;
    }

    public function getViewFile()
    {
        return Madhouse_ThemeKitchen_Package::getPath("views/admin/_tab-design.php");
    }

    public function getId()
    {
        return "design";
    }

    /**
     * Compile the given $scss to css.
     *
     * @return string
     */
    public function generateCss($compiler, $scssAsString, $variables)
    {
        // Get the compiler object.
        $scssCompiler = $compiler;
        $scssCompiler->setVariables($variables);

        try {
            // Compile to css.
            $cssAsString = $scssCompiler->compile($scssAsString);
        } catch (ParserException $e) {
            throw new Exception($e->getMessage(), 0, $e);
        } catch (CompilerException $e) {
            throw new Exception($e->getMessage(), 1, $e);
        } catch (ServerException $e) {
            throw new Exception($e->getMessage(), 2, $e);
        }

        // Return the generated css code.
        return $cssAsString;
    }

    /**
     * {@inheritdoc}
     */
    public function update($settings)
    {
        // Get styles configuration.
        $styles = $this->configuration->get("$.styles");
        if (!$styles) {
            // No styles configuration. No compilation.
            return;
        }

        // Get custom subsection.
        $subsections = $settings->get("custom");

        // Concat all scss files.
        $scss = "";
        $variables = array();
        foreach ($subsections as $key => $custom) {
            $nonEmptyVariables = array_filter($custom["variables"]);
            if (count($nonEmptyVariables) === count($custom["variables"])) {
                // Get the content of the file.
                $rawScss = $this->readFile($this->themes->getCurrentThemePath() . $custom["path"]);
                if ($rawScss) {
                    // If file is not empty.
                    $scss .= $rawScss;
                    $variables = array_merge($variables, $nonEmptyVariables);
                }
            }
        }

        $customCss = "";
        if ($scss !== "") {
            // Custom scss is generated to css code.
            $customCss = $this->generateCss(
                $this->getCompiler($styles["type"], $styles["options"]),
                $scss,
                $variables
            );
        }

        // Get current (or newly selected variant)
        $variant = $settings->get("$", "s_variant");

        // Get source and dest path.
        $sourcePath = $this->themes->getCurrentThemePath() . $styles["path"];
        $destPath = $this->themes->getCurrentThemePath() . $styles["path"];

        // Get base css code from source path.
        $baseCss = file_get_contents($sourcePath . "/" . $variant . ".min.css");

        // Get custom css (vanilla code in the admin)
        $adminCss = $settings->get("$", "custom_css");

        // Write to filesystem.
        $this->writeFile($destPath . "/overwrite.min.css", array($customCss, "\n", $adminCss));
        $this->writeFile($destPath . "/main.min.css", array($baseCss, "\n", $customCss, "\n", $adminCss));
    }

    /**
     * {@inheritdoc}
     */
    public function refresh($settings)
    {
        return $this->update($settings);
    }

    public function getCompiler($type, $options = null)
    {
        if (is_null($options)) {
            $options = array();
        }

        if ($type === "scss") {
            // Create a new scss compiler.
            $scssCompiler = new ScssCompiler();

            // Handle include paths.
            if (
                isset($options["include_paths"])
                && count($options["include_paths"]) > 0
            ) {
                // Resolve each include path.
                $that = $this;
                $includePaths = array_map(
                    function ($relativePath) use ($that) {
                        return $that->themes->getCurrentThemePath() . $relativePath;
                    },
                    $options["include_paths"]
                );

                // Set include paths.
                $scssCompiler->setImportPaths($includePaths);
            }

            // Return compiler.
            return $scssCompiler;
        }

        // No compiler found.
        throw new Exception("No compiler found for type '" . $type . "'", 5);
    }

    /**
     * Read the content of a file located at $path.
     *
     * @param  string $path
     *
     * @return string
     */
    protected function readFile($path)
    {
        $content = @file_get_contents($path);
        if ($content === false) {
            // Throw error if one scss does not exits, like @import would.
            $error = error_get_last();
            throw new Exception($error['message'], 4);
        }
        return $content;
    }

    /**
     * Write the content $content to a file $path.
     *
     * @param  string $path
     * @param  string $content
     *
     * @return void
     */
    protected function writeFile($path, $content)
    {
        $isWritten = @file_put_contents($path, $content);
        if ($isWritten === false) {
            $error = error_get_last();
            throw new Exception($error['message'], 3);
        }
    }
}
